/*
	webfiltd 0.96 - Daemon component
	Copyright (C) 2004, Jem E. Berkes <jberkes@pc-tools.net>
	http://www.pc-tools.net/unix/webfilt/
	
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#define _GNU_SOURCE
#include <ctype.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <pwd.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "md5.h"
#include "webfilt.h"
#include "config.h"
#ifdef USE_SHADOW_H
#include <shadow.h>
#endif

/* Function prototypes */
int read_conf();
int read_userconf(const char*);
void logger(int, const char*, const char*);
int login(const char*, const char*);
int relogin(const char*, const char*);
struct passwd* unix_auth(const char*, const char*);
int gen_secret(char*);
void gen_hash(const char*, const char*, char*);
int save_session(const char*);
int valid_session(const char*);
int safe_parm(const char*);
void fix_environment(const char*, const char*);
void session_clean();

/* Global variables */
struct daemonconf daemonconfig;
struct userconf userconfig;
char peer_ip[BUFSIZE];


int main()
{
	struct sockaddr_in sin_peer;
	int sin_len = sizeof(sin_peer);
	struct stat dirstat;
	char linebuf[BUFSIZE], username[BUFSIZE], password[BUFSIZE];
	unsigned int folder;	/* 0=good, 1=spam */
	char *dirfolder, *dirother;
	struct passwd* lowpriv;

	/* Initialize vital buffers */
	memset(&daemonconfig, 0, sizeof(daemonconfig));
	memset(&userconfig, 0, sizeof(userconfig));
	memset(peer_ip, 0, BUFSIZE);

	/* Get IP address of connecting host */
	if (getpeername(0, (struct sockaddr*)&sin_peer, (socklen_t *)&sin_len) == 0)
		strncpy(peer_ip, inet_ntoa(sin_peer.sin_addr), BUFSIZE-1);
	else
		strcpy(peer_ip, "unknown");

	/* Read and parse configuration file */
	if (!read_conf())
		return RETCODE_FAIL;
		
	/* Verify that my IP is connecting (service for local use only) */
	if (strcmp(daemonconfig.myip, peer_ip) != 0)
	{
		logger(LOG_WARNING, "Not from myip", daemonconfig.myip);
		return RETCODE_FAIL;
	}
		
	/* Verify safe mode on sessiondir */
	if (stat(daemonconfig.sessiondir, &dirstat) == -1)
	{
		logger(LOG_WARNING, "Can't stat()", daemonconfig.sessiondir);
		return RETCODE_FAIL;
	}
	if (dirstat.st_mode != (S_IFDIR | S_IRWXU))
	{
		logger(LOG_WARNING, "Unsafe permissions on", daemonconfig.sessiondir);
		return RETCODE_FAIL;
	}
	
	/* Change working directory to sessiondir */
	if (chdir(daemonconfig.sessiondir) != 0)
	{
		logger(LOG_WARNING, "Can't chdir()", daemonconfig.sessiondir);
		return RETCODE_FAIL;
	}
	
	/* Change effective privileges */
	lowpriv = getpwnam(daemonconfig.user);
	if (!lowpriv)
	{
		logger(LOG_WARNING, "Invalid daemon user", daemonconfig.user);
		return RETCODE_FAIL;
	}
	if ((setegid(lowpriv->pw_gid) != 0) || (seteuid(lowpriv->pw_uid) != 0))
	{
		logger(LOG_WARNING, "Unable to change to user", daemonconfig.user);
		return RETCODE_FAIL;
	}
	
	/* Get authentication command */
	if (!fgets(linebuf, sizeof(linebuf), stdin))
	{
		logger(LOG_WARNING, "Unable to read command", NULL);
		return RETCODE_FAIL;
	}

	if (sscanf(linebuf, "LOGIN %s %s %u", username, password, &folder) == 3)
	{
		if (!login(username, password))
		{
			printf("-LOGIN failed\n");
			fflush(stdout);
			return RETCODE_FAIL;
		}
	}
	else if (sscanf(linebuf, "RELOGIN %s %s %u", username, password, &folder) == 3)
	{
		if (!relogin(username, password))
		{
			printf("-RELOGIN failed\n");
			fflush(stdout);
			return RETCODE_FAIL;
		}
	}
	else
	{
		printf(MSG_BADCMD);
		fflush(stdout);
		logger(LOG_WARNING, "Invalid authentication command", NULL);
		return RETCODE_FAIL;
	}
	
	/* User logged on */
	if (!read_userconf(username))
		return RETCODE_FAIL;
		
	/* Switch to dirgood or dirspam, according to login folder */
	if (folder)
	{
		dirfolder = userconfig.dirspam;
		dirother = userconfig.dirgood;
	}
	else
	{
		dirfolder = userconfig.dirgood;
		dirother = userconfig.dirspam;
	}
		
	if (chdir(dirfolder) != 0)
	{
		logger(LOG_WARNING, "Can't set dir.folder for user", username);
		return RETCODE_FAIL;
	}

	/* Enter user-based command handling loop */
	while (fgets(linebuf, sizeof(linebuf), stdin))
	{
		char parm1[BUFSIZE], parm2[BUFSIZE];
		if (strncmp(linebuf, "QUIT", 4) == 0)
			return RETCODE_OK;
		else if (strncmp(linebuf, "LISTEXEC", 8) == 0)
		{
			int pos;
			for (pos=0; pos<userconfig.cmdcount; pos++)
				printf("%s\n", userconfig.exec[pos].label);
			printf(MSG_OK);
			fflush(stdout);
		}	/* LISTEXEC */
		else if (sscanf(linebuf, "EXEC %s %s", parm1, parm2) == 2)
		{
			int ran=0, pos;
			if (!safe_parm(parm2))
				continue;			
			for (pos=0; !ran && (pos<userconfig.cmdcount); pos++)
			{
				if (strcmp(parm1, userconfig.exec[pos].label) == 0)
				{
					FILE* cmd;
					char cmdline[2*BUFSIZE];
					sprintf(cmdline, "%s %s", userconfig.exec[pos].param, parm2);
					cmd = popen(cmdline, "r");
					if (cmd)
					{
						while (fgets(linebuf, sizeof(linebuf), cmd))
							printf("%s", linebuf);
						printf("\n");
						ran = 1;
						pclose(cmd);
					}
				}
			}
			if (ran)
			{
				printf(MSG_OK);
				fflush(stdout);
			}
			else
			{
				printf(MSG_BADCMD);
				fflush(stdout);
			}
		}	/* EXEC */
		else if (strncmp(linebuf, "LIST", 4) == 0)
		{
			FILE* lscmd = popen(LIST_CMD, "r");
			if (lscmd)
			{
				while (fgets(linebuf, sizeof(linebuf), lscmd))
					printf("%s", linebuf);
				pclose(lscmd);
			}
			else
				logger(LOG_WARNING, "Unable to LIST for user", username);
			printf(MSG_OK);
			fflush(stdout);
		}	/* LIST */
		else if ( (sscanf(linebuf, "HEAD %s", parm1) == 1) || (sscanf(linebuf, "GET %s", parm1) == 1))
		{
			FILE* msgfile;
			int head = (*linebuf == 'H');
			if (!safe_parm(parm1))
				continue;
			msgfile = fopen(parm1, "r");
			if (msgfile)
			{
				while (fgets(linebuf, sizeof(linebuf), msgfile))
				{
					if (head && ((*linebuf == '\r') || (*linebuf == '\n')))
							break;
					printf("%s", linebuf);
				}
				printf("\n");
				fclose(msgfile);
				printf(MSG_OK);
			}
			else
				printf(MSG_BADFILE);
			fflush(stdout);
		}	/* GET, HEAD */
		else if (sscanf(linebuf, "SWAP %s", parm1) == 1)
		{
			char destpath[2*BUFSIZE];
			if (!safe_parm(parm1))
				continue;
			sprintf(destpath, "%s/%s", dirother, parm1);
			if (rename(parm1, destpath) == 0)
				printf(MSG_OK);
			else
				printf(MSG_BADFILE);
			fflush(stdout);
		}	/* SWAP */
		else
		{
			printf(MSG_BADCMD);
			fflush(stdout);
		}
	}
	return RETCODE_FAIL;
}


/*
	Reads and parses the daemon's configuration file
	Returns 0 on failure with explanation to syslog
	Returns 1 on success

*/
int read_conf()
{
	char linebuf[BUFSIZE];
	FILE* config = fopen(CONF_FILE, "r");
	if (!config)
	{
		logger(LOG_WARNING, "Can't open", CONF_FILE);
		return 0;
	}
	while (fgets(linebuf, sizeof(linebuf), config))
	{
		if (sscanf(linebuf, "myip %s", daemonconfig.myip) == 1)
			continue;
		if (sscanf(linebuf, "user %s", daemonconfig.user) == 1)
			continue;
		if (sscanf(linebuf, "sessiondir %s", daemonconfig.sessiondir) == 1)
			continue;
	}
	fclose(config);

	if (daemonconfig.myip[0] && daemonconfig.user[0] && daemonconfig.sessiondir[0])
		return 1;
	else
	{
		logger(LOG_WARNING, ".conf syntax error", NULL);
		return 0;
	}
}


/*
	Reads and parses the user's configuration file
	Returns 0 on failure with explanation to syslog
	Returns 1 on success
*/
int read_userconf(const char* username)
{
	char linebuf[BUFSIZE];
	FILE* config = fopen(USERCONF_FILE, "r");
	if (!config)
	{
		logger(LOG_WARNING, "Can't open " USERCONF_FILE " for user", username);
		return 0;
	}
	while (fgets(linebuf, sizeof(linebuf), config))
	{
		if (sscanf(linebuf, "dir.good %s", userconfig.dirgood) == 1)
			continue;
		if (sscanf(linebuf, "dir.spam %s", userconfig.dirspam) == 1)
			continue;
		if ((userconfig.cmdcount < MAXEXEC) && (sscanf(linebuf, "exec.%s %[^\r\n]",
			userconfig.exec[userconfig.cmdcount].label, userconfig.exec[userconfig.cmdcount].param) == 2))
		{
			userconfig.cmdcount++;
			continue;
		}
	}
	fclose(config);

	if (userconfig.dirgood[0] && userconfig.dirspam[0] && userconfig.cmdcount)
		return 1;
	else
	{
		logger(LOG_WARNING, USERCONF_FILE " syntax error for user", username);
		return 0;
	}
}




void logger(int priority, const char* msg1, const char* msg2)
{
	openlog("webfiltd", LOG_PID, LOG_DAEMON);
	if (msg2)	
		syslog(priority, "[%s] %s %s", peer_ip, msg1, msg2);
	else
		syslog(priority, "[%s] %s", peer_ip, msg1);
	closelog();
}


/*
	Attempts a new login (also sets user, and changes to home dir)
	Causes of failure logged via syslog
	
	Returns 0 on failure
	Returns 1 on success
*/
int login(const char* username, const char* password)
{
	struct passwd* newuser;
	char secret[SECRETLEN+1];
	char hash_output[BUFSIZE];
	seteuid(0);
	setegid(0);
	newuser = unix_auth(username, password);
	if (newuser == NULL)
	{
		logger(LOG_WARNING, "LOGIN failed for", username);
		return 0;
	}
	/* Authentication successful */	
	memset(secret, 0, sizeof(secret));
	if (gen_secret(secret) < MINSECRETLEN)
	{
		logger(LOG_WARNING, "Can't generate secret for", username);
		return 0;
	}
	/* We now have a random secret, make session hash */
	gen_hash(username, secret, hash_output);
	session_clean();
	if (!save_session(hash_output))
	{
		logger(LOG_WARNING, "Can't save session for", username);
		return 0;
	}

	/* Authenticated, session recorded. Switch to new user */
	if ((setgid(newuser->pw_gid) != 0) || (setuid(newuser->pw_uid) != 0))
	{
		logger(LOG_WARNING, "Can't set user", username);
		return 0;
	}
	
	/* Set user's home directory */
	if (chdir(newuser->pw_dir) != 0)
	{
		logger(LOG_WARNING, "Can't set homedir for user", username);
		return 0;
	}
	fix_environment(username, newuser->pw_dir);

	printf("+%s\n", secret);	/* tell client the secret */
	fflush(stdout);
	logger(LOG_NOTICE, "LOGIN", username);
	return 1;		/* successful completion of LOGIN */
}


/*
	Attempts relogin (also sets user, and changes to home dir)
	Causes of failure logged via syslog
	
	Returns 0 on failure
	Returns 1 on success
*/
int relogin(const char* username, const char* secret)
{
	char hash_output[BUFSIZE];
	struct passwd* newuser;
	seteuid(0);
	setegid(0);
	gen_hash(username, secret, hash_output);
	if (!valid_session(hash_output))
	{
		logger(LOG_WARNING, "RELOGIN failed, invalid session for", username);
		return 0;
	}
	
	/* The client knew the username and matching secret */
	newuser = getpwnam(username);
	if (newuser == NULL)
	{
		logger(LOG_WARNING, "Can't RELOGIN invalid user", username);
		return 0;
	}
	
	/* Session verified and user exists. Switch to returning user */
	if ((setgid(newuser->pw_gid) != 0) || (setuid(newuser->pw_uid) != 0))
	{
		logger(LOG_WARNING, "Can't set user", username);
		return 0;
	}

	/* Set user's home directory */
	if (chdir(newuser->pw_dir) != 0)
	{
		logger(LOG_WARNING, "Can't set homedir for user", username);
		return 0;
	}
	fix_environment(username, newuser->pw_dir);

	printf(MSG_OK);
	fflush(stdout);
	/* logger(LOG_NOTICE, "RELOGIN", username); */
	return 1;		/* successful completion of LOGIN */
}


/*
	Attempts authentication of given *NIX account
	Returns 0 on failure (invalid username/password)
	Returns uid on success (authenticated)
*/
struct passwd* unix_auth(const char* username, const char* password)
{
	struct passwd* pwentry = getpwnam(username);
	if (pwentry && (pwentry->pw_uid > 1) && (strlen(pwentry->pw_passwd) > 0))
	{
		if (strcmp(pwentry->pw_passwd, crypt(password, pwentry->pw_passwd)) == 0)
		{
			/* correct password, classic crypt or transparent BSD shadow */
			return pwentry;
		}
		#ifdef USE_SHADOW_H
		else
		{
			struct spwd* shdwentry = getspnam(pwentry->pw_name);
			if (shdwentry && (strlen(shdwentry->sp_pwdp) > 0))
			{
				if (strcmp(shdwentry->sp_pwdp, crypt(password, shdwentry->sp_pwdp)) == 0)
				{
					/* correct password, explicit shadow (Linux, Solaris) */
					return pwentry;
				}
			}
		}
		#endif
	}
	return NULL;	/* catch-all failure */
}


/*
	Generate ASCII secret using high-entropy random device
	Destination buffer must be entirely cleared (SECRETLEN+1)
	Returns 0 on failure
	Returns length of generating string on success
*/
int gen_secret(char* secret)
{
	FILE* randsource = fopen(RAND_FILE, "rb");
	if (randsource)
	{
		int got, dstpos=0;
		while ((dstpos<SECRETLEN) && ((got=fgetc(randsource))!=EOF))
		{
			if (isalnum(got))
				secret[dstpos++] = (char)got;
		}
		fclose(randsource);
		return strlen(secret);
	}
	return 0;
}


void gen_hash(const char* username, const char* secret, char* hash_output)
{
	struct md5_ctx md5context;
	unsigned char md5digest[16];
	char hash_input[BUFSIZE+SECRETLEN];
	int pos;
	strcpy(hash_input, username);
	strcat(hash_input, secret);
	md5_init_ctx(&md5context);
	md5_process_bytes(hash_input, strlen(hash_input), &md5context);
	md5_finish_ctx(&md5context, md5digest);
	for (pos=0; pos<16; pos++)
		sprintf(hash_output+(pos*2), "%02x", md5digest[pos]);
}


/*
	Saves a session file containing creation timestamp
	Returns 1 on success
	Returns 0 on failure
*/
int save_session(const char* filename)
{
	FILE* session = fopen(filename, "w");
	if (session)
	{
		fprintf(session, "%u\n", (unsigned int)time(NULL));
		fclose(session);
		return 1;
	}
	else
		return 0;
}


/*
	Checks if this is a valid session
	Returns 1 on success
	Returns 0 on failure
*/
int valid_session(const char* filename)
{
	FILE* session = fopen(filename, "r");
	if (session)
	{
		unsigned int timestamp, timenow = (unsigned int) time(NULL); 
		if (fscanf(session, "%u", &timestamp) == 1)
		{
			if ((timenow > timestamp) && (timenow-timestamp < SESSION_LEN))
			{
				fclose(session);
				return 1;
			}
		}
		fclose(session);
	}
	return 0;
}


/*
	Tests given parameter for '$', '~', '/' and '..' (disallowed)

	Returns 0 if not safe
	Returns 1 if safe
*/
int safe_parm(const char* parm)
{
	if (strchr(parm, '$') || strchr(parm, '~') || strchr(parm, '/') || strstr(parm, ".."))
	{
		printf(MSG_BADPARM);
		fflush(stdout);
		return 0;
	}
	else
		return 1;
}


/*
	Change some vital environment variables
*/
void fix_environment(const char* username, const char* homedir)
{
	setenv("HOME", homedir, 1);
	setenv("PWD", homedir, 1);
	setenv("USER", username, 1);
}


/*
	Cleanup expired sessions, periodically
	Must already be in session directory
*/
void session_clean()
{
	unsigned int timenow = (unsigned int) time(NULL);
	struct dirent* entry;
	DIR* sessiondir;
	if (getpid() % 3 != 0)
		return;
		
	sessiondir = opendir(".");
	if (!sessiondir)
	{
		logger(LOG_WARNING, "Unable to list session directory (cleanup)", NULL);
		return;
	}
	
	entry = readdir(sessiondir);
	while (entry)
	{
		if (*entry->d_name != '.')
		{
			FILE* session = fopen(entry->d_name, "r");
			if (session)
			{
				int expired = 0;
				unsigned int timestamp;
				if (fscanf(session, "%u", &timestamp) == 1)
				{
					if ((timenow < timestamp) || (timenow-timestamp > SESSION_LEN))
						expired = 1;
				}
				else
					logger(LOG_WARNING, "Error reading timestamp (cleanup)", entry->d_name);
				fclose(session);
				if (expired)
				{
					if (unlink(entry->d_name) != 0)
						logger(LOG_WARNING, "Error deleting (cleanup)", entry->d_name);
				}
			}
			else
				logger(LOG_WARNING, "Error opening (cleanup)", entry->d_name);
		}
		entry = readdir(sessiondir);
	}
	closedir(sessiondir);
}
