/*
	SpamTestBuddy 0.94

	Copyright (C) 2007 Jem E. Berkes
	http://www.sysdesign.ca/
	
	This software may be used freely for personal, non-commercial use and
	educational use. It may not be re-distributed, re-sold, included in a
	commercial package or modified without the author's permission.
	
	
	Configuration is loaded from $HOME/.spamtestbuddy
	
	Returns 0 on success (configuration loaded, email processed) with result to stdout
	Returns 1 on failure (bad configuration) with errors to stderr
	
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "spamtestbuddy.h"

#define VERSION		"0.94"

/* #define VERBOSE if you want some useful debugging output to stderr */

#define SCOREUNIT	1.0


void process_headers();
int test_dns_problems(const char* ipstr);
void test_file_ip(const char* ipstr);
void test_header_substr(const char* linebuf);
void test_header_yes(const char* linebuf);
void test_header_float(const char* linebuf);
void test_dnsbl(unsigned int* ip);
void output_test_results(const char* ipstr);
void show_matches(struct testentry* pos);
int ip_is_local(const char* ipstr);
int load_configuration(FILE* config);
int add_test(struct testentry** chain, const char* prefix, const char* testname, const char* param);
void show_config();

/* Global variables (program instance handles one email) */
float score = 0;

/* Loaded from configuration */
float threshold = 0;				/* SpamThreshold */
char local_hops[MAXLINEBUF] = "";		/* SkipReceived */
struct testentry* testdnsproblems = NULL;	/* TestDnsProblems */
struct testentry* testfileip = NULL;		/* TestFileIP */
struct testentry* testheadersubstr = NULL;	/* TestHeaderSubstr */
struct testentry* testheaderyes = NULL;		/* TestHeaderYes */
struct testentry* testheaderfloat = NULL;	/* TestHeaderFloat */
struct testentry* testdnsbl = NULL;		/* TestDNSBL */


int main()
{
	FILE* configfile;
	char configfn[MAXLINEBUF];
	char* homedir;
	
	homedir = getenv("HOME");
	if (!homedir)
	{
		fprintf(stderr, "Can't load configuration, HOME undefined\n");
		return 1;
	}
	snprintf(configfn, sizeof(configfn), "%s/.spamtestbuddy", homedir);
	configfile = fopen(configfn, "r");
	if (!configfile)
	{
		perror(configfn);
		return 1;
	}
	if (load_configuration(configfile) == 0)
	{
		/* ready to go, process email at stdin */
		char linebuf[MAXLINEBUF];
		fclose(configfile);
		#ifdef VERBOSE
		show_config();
		#endif
		process_headers();
		/* spit out the rest of the email */
		while (fgets(linebuf, sizeof(linebuf), stdin))
			printf("%s", linebuf);
		return 0;
	}
	else
	{
		fprintf(stderr, "load_configuration() failed, aborting\n");
		fclose(configfile);
		return 1;
	}
}


/*
	Read email input from stdin until the end of email headers.
	Parse out the sending IP address from Received headers
*/
void process_headers()
{
	int got_ip = 0;
	unsigned int ip[4] = {0, 0, 0, 0};	/* sender's IP address */
	char linebuf[MAXLINEBUF], ipstr[MAXLINEBUF] = "none";
	while (fgets(linebuf, sizeof(linebuf), stdin))
	{
		if (*linebuf == '\n')
		{
			/* Reached end of headers */
			/* TestDnsProblems */
			if (got_ip && testdnsproblems && test_dns_problems(ipstr))
			{
				testdnsproblems->match = 1;	/* yes, there is a DNS problem */
				if (testdnsproblems->adds)
					score += SCOREUNIT;
				else
					score -= SCOREUNIT;
			}
			/* TestFileIP */
			if (got_ip)
				test_file_ip(ipstr);	/* scores applied in function */
			/* TestDNSBL */
			if (got_ip)
				test_dnsbl(ip);		/* scores applied in function */
			
			output_test_results(ipstr);	/* if haven't got_ip uses default ipstr=none */
			return;
		}
		else if (!got_ip && (strncmp(linebuf, "Received: from ", 15)==0))
		{
			char* bracket = strchr(linebuf, '[');
			if (bracket)
			{
				if ((sscanf(bracket+1, "%u.%u.%u.%u", &ip[0], &ip[1], &ip[2], &ip[3]) == 4) &&
				(ip[0]<256) && (ip[1]<256) && (ip[2]<256) && (ip[3]<256))
				{
					sprintf(ipstr, "%u.%u.%u.%u", ip[0], ip[1], ip[2], ip[3]);
					if (ip_is_local(ipstr))
						strcpy(ipstr, "none");
					else
					{
						got_ip = 1;
						#ifdef VERBOSE
						fprintf(stderr, "Parsed sender IP %s\n", ipstr);
						#endif
					}
				}
			}
		}
		else
		{
			/* For the tests on headers, scoring is done in the test_ function */
			test_header_substr(linebuf);	/* TestHeaderSubstr */
			test_header_yes(linebuf);	/* TestHeaderYes */
			test_header_float(linebuf);	/* TestHeaderFloat */
		}
		printf("%s", linebuf);	/* output the line back to stdout */
	}
}




/*
	If this header line matches, the appropriate testentry and score is updated
*/
void test_header_substr(const char* linebuf)
{
	struct testentry* pos = testheadersubstr;
	while (pos)
	{
		if (!(pos->match))
		{
			if (strncmp(linebuf, pos->param, strlen(pos->param)) == 0)
			{
				pos->match = 1;
				if (pos->adds)
					score += SCOREUNIT;
				else
					score -= SCOREUNIT;
			}
		}
		pos = pos->next;
	}
}



/*
	If this header line matches, the appropriate testentry and score is updated
*/
void test_header_yes(const char* linebuf)
{
	struct testentry* pos = testheaderyes;
	while (pos)
	{
		if (!(pos->match))
		{
			if (strncmp(linebuf, pos->param, strlen(pos->param)) == 0)
			{
				char word[MAXLINEBUF]="";
				/* Found the right line */
				if ((sscanf(linebuf+strlen(pos->param), "%s", word) == 1) &&
				((strcasecmp(word, "yes")==0)||(strcasecmp(word, "true")==0)||(strcasecmp(word, "spam")==0)))
				{
					pos->match = 1;
					if (pos->adds)
						score += SCOREUNIT;
					else
						score -= SCOREUNIT;
				}
			}
		}
		pos = pos->next;
	}
}

/*
	If the header containing the number is found, update score and testentry
*/
void test_header_float(const char* linebuf)
{
	struct testentry* pos = testheaderfloat;
	while (pos)
	{
		if (!(pos->match))
		{
			if (strncmp(linebuf, pos->param, strlen(pos->param)) == 0)
			{
				float value = 0;
				/* Start parsing on the number, which must be 0 - 1.0001 */
				if ((sscanf(linebuf+strlen(pos->param), "%*[^0-9.]%f", &value) == 1)
					&& (value >= 0) && (value < 1.0001))
				{
					pos->match = 1;
					if (pos->adds)
						score += value;
					else
						score -= value;
				}
			 }
		}
		pos = pos->next;
	}
}


/*
	Test IP against files containing IP lists
	The function is called once per email
*/
void test_file_ip(const char* ipstr)
{
	struct testentry* pos = testfileip;
	while (pos)
	{
		FILE* localfile = fopen(pos->param, "r");
		if (localfile)
		{
			char fileline[MAXLINEBUF];
			while (!pos->match && fgets(fileline, sizeof(fileline), localfile))
			{
				char* end = strchr(fileline, '\r');
				if (!end) end = strchr(fileline, '\n');
				if (end) *end = '\0';
				/* do exact comparison */
				if (strcmp(ipstr, fileline) == 0)
				{
					pos->match = 1;
					if (pos->adds)
						score += SCOREUNIT;
					else
						score -= SCOREUNIT;
				}
			}
			fclose(localfile);
		}
		else
			perror(pos->param);	/* error to stderr if could not open file */
		
		pos = pos->next;
	}
}


/*
	Test all DNSBLs, updating scores
	The function is called once per amil
*/
void test_dnsbl(unsigned int* ip)
{
	struct testentry* pos = testdnsbl;
	while (pos)
	{
		struct hostent* h;
		char hostname[MAXLINEBUF*2];
		/* construct reverse octet format hostname to look up under zone */
		sprintf(hostname, "%u.%u.%u.%u.%s", ip[3], ip[2], ip[1], ip[0], pos->param);
		h = gethostbyname(hostname);
		if (h && (h->h_addr_list) && (h->h_addr_list[0]))
		{
			if (strncmp(inet_ntoa(*(struct in_addr*)(h->h_addr_list[0])), "127.0.", 6) == 0)
			{
				pos->match = 1;
				if (pos->adds)
					score += SCOREUNIT;
				else
					score -= SCOREUNIT;
			}
			/* keep processing, check all DNSBLs for this IP */
		}
		pos = pos->next;
	}
}




/*
	Test that
	1. The IP address resolves to a host name (reverse dns)
	2. The host name has this IP as an address record (forward dns)
	
	If both forward and reverse dns are fine, then returns 0
	If there is a problem (match), returns non-zero
*/
int test_dns_problems(const char* ipstr)
{
	struct in_addr a;
	memset(&a, 0, sizeof(a));
	a.s_addr = inet_addr(ipstr);
	if (a.s_addr == INADDR_NONE)
		return 1;	/* not valid IP address */
	else
	{
		struct hostent* h = gethostbyaddr((char*)&a, sizeof(a), AF_INET);
		if (h && (h->h_name))	/* check for reverse dns */
		{
			char hostname[MAXLINEBUF], *newipstr=NULL;
			memset(hostname, 0, sizeof(hostname));
			strncpy(hostname, h->h_name, sizeof(hostname)-1);
			/* might it be fake? */
			h = gethostbyname(hostname);	/* resolve A records for host */
			if (h && (h->h_addr_list))
			{
				int i;
				for (i=0; (h->h_addr_list[i]); i++)
				{
					newipstr = inet_ntoa(*(struct in_addr*)(h->h_addr_list[i]));
					if (newipstr && (strcmp(newipstr, ipstr)==0))
					{
						/* both reverse and matching forward dns exist */
						return 0;
					}
				}
			}
		}
	}
	return 1;		/* did not succeed for various reasons (no reverse DNS, no forward DNS, inconsistent) */
}


int ip_is_local(const char* ipstr)
{
	char networks[MAXLINEBUF];
        char* tok = NULL;
        strncpy(networks, local_hops, sizeof(networks));
	tok = strtok(networks, LIST_TOKENS);
	while (tok)
	{
		if (strncmp(tok, ipstr, strlen(tok)) == 0)
			return 1;	/* ipstr matches a network in the list */
		tok = strtok(NULL, LIST_TOKENS);
	}
	return 0;			/* ipstr does not match any */
}


/*
	Show results of all tests done (scores already applied)
*/
void output_test_results(const char* ipstr)
{
	const char* label;
	if (score >= threshold)
		label = "OVER";
	else
		label = "UNDER";
	printf("X-SpamTestBuddy: %s score=%.4f threshold=%.4f ip=%s\n", label, score, threshold, ipstr);
	printf("X-SpamTestBuddy-Tests: ver=" VERSION " ");
	/* Show testnames that go along with all tests which matched */
	show_matches(testdnsproblems);
	show_matches(testfileip);
	show_matches(testheadersubstr);
	show_matches(testheaderyes);
	show_matches(testheaderfloat);
	show_matches(testdnsbl);
	puts("\n");
}


void show_matches(struct testentry* pos)
{
	for (; pos; pos=pos->next)
	{
		if (pos->match)
			printf("%s ", pos->testname);
	}	
}


/*
	Load, parse the configuration file
	Return nonzero and errors to stderr if there are errors
*/
int load_configuration(FILE* config)
{
	char linebuf[MAXLINEBUF];
	while (fgets(linebuf, sizeof(linebuf), config))
	{
		float number;
		char prefix[MAXLINEBUF];
		char testname[MAXLINEBUF];
		char param[MAXLINEBUF];

		if ((*linebuf=='#') || (*linebuf=='\r') || (*linebuf=='\n'))
			continue;

		if (sscanf(linebuf, "SpamThreshold %f", &number) == 1)
			threshold = number;
		else if (sscanf(linebuf, "SkipReceived %[^\r\n]", param) == 1)
			strncpy(local_hops, param, sizeof(local_hops));
		else if (sscanf(linebuf, "%[+-]TestDnsProblems %s", prefix, testname) == 2)
		{
			if (add_test(&testdnsproblems, prefix, testname, ""))
				return 1;
		}
		else if (sscanf(linebuf, "%[+-]TestFileIP %s %s", prefix, testname, param) == 3)
		{
			if (add_test(&testfileip, prefix, testname, param))
				return 1;
		}
		else if (sscanf(linebuf, "%[+-]TestHeaderSubstr %s %[^\r\n]", prefix, testname, param) == 3)
		{
			if (add_test(&testheadersubstr, prefix, testname, param))
				return 1;
		}
		else if (sscanf(linebuf, "%[+-]TestHeaderYes %s %s", prefix, testname, param) == 3)
		{
			if (add_test(&testheaderyes, prefix, testname, param))
				return 1;
		}
		else if (sscanf(linebuf, "%[+-]TestHeaderFloat %s %s", prefix, testname, param) == 3)
		{
			if (add_test(&testheaderfloat, prefix, testname, param))
				return 1;
		}
		else if (sscanf(linebuf, "%[+-]TestDNSBL %s %s", prefix, testname, param) == 3)
		{
			if (add_test(&testdnsbl, prefix, testname, param))
				return 1;
		}
		else
		{
			fprintf(stderr, "Bad configuration line: %s", linebuf);
			return 1;
		}
	}
	return 0;
}


/*
	Returns 0 on success, nonzero on failure
*/
int add_test(struct testentry** chain, const char* prefix, const char* testname, const char* param)
{
	struct testentry* newtest = calloc(1, sizeof(struct testentry));
	struct testentry* pos = *chain;
	
	if (*prefix == '+')
		newtest->adds = 1;
	else if (*prefix == '-')
		newtest->adds = 0;
	else
	{
		fprintf(stderr, "Missing +/- in configuration test\n");
		return 1;
	}
	/* Modify testname to include + or - for informative purposes */
	newtest->testname[0] = *prefix;
	strncpy(newtest->testname+1, testname, MAXLINEBUF-1);
	strncpy(newtest->param, param, MAXLINEBUF);
	
	if (pos == NULL)	/* create chain, first entry */
	{
		*chain = newtest;
		pos = newtest;
	}
	else
	{
		while (pos->next)
			pos = pos->next;
		pos->next = newtest;
	}
	return 0;
}


#ifdef VERBOSE
void show_config()
{
	struct testentry* pos;
	char networks[MAXLINEBUF];
	char* tok = NULL;
	strncpy(networks, local_hops, sizeof(networks));
	fprintf(stderr, "SpamTestBuddy " VERSION "\n\n");
	fprintf(stderr, "SpamThreshold = %f\n", threshold);
	fprintf(stderr, "SkipReceived =\n");
	tok = strtok(networks, LIST_TOKENS);
	while (tok)
	{
		fprintf(stderr, "\t%s\n", tok);
		tok = strtok(NULL, LIST_TOKENS);
	}
	fprintf(stderr, "\nadd/sub\ttest\t\tname\tparam\n\n");
	if (testdnsproblems)
		fprintf(stderr, "%d\tTestDnsProblems\t%s\tN/A\n", testdnsproblems->adds, testdnsproblems->testname);
	for (pos=testfileip; pos; pos=pos->next)
		fprintf(stderr, "%d\tTestFileIP\t%s\t%s\n", pos->adds, pos->testname, pos->param);
	for (pos=testheadersubstr; pos; pos=pos->next)
		fprintf(stderr, "%d\tTestHeaderSubstr\t%s\t%s\n", pos->adds, pos->testname, pos->param);
	for (pos=testheaderyes; pos; pos=pos->next)
		fprintf(stderr, "%d\tTestHeaderYes\t%s\t%s\n", pos->adds, pos->testname, pos->param);
	for (pos=testheaderfloat; pos; pos=pos->next)
		fprintf(stderr, "%d\tTestHeaderFloat\t%s\t%s\n", pos->adds, pos->testname, pos->param);
	for (pos=testdnsbl; pos; pos=pos->next)
		fprintf(stderr, "%d\tTestDNSBL\t%s\t%s\n", pos->adds, pos->testname, pos->param);
}

#endif
