/*
	smtpauth 0.94 - Authenticating proxy for servers without SMTP AUTH
	Copyright (C) 2004, Jem E. Berkes <jberkes@pc-tools.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#define AUTHFILE	"/etc/smtpauth.conf"
#define BUFSIZE	4096

#define ERROR_COMMANDARGS	1
#define ERROR_BADIPADDRESS	2
#define ERROR_NOSOCKET		3
#define ERROR_FAILCONNECT	4
#define ERROR_READFAILURE	5
#define ERROR_NOPERMISSION	6
#define ERROR_LOSTSOCKET	7
#define ERROR_NOAUTHFILE	8
#define ERROR_UIDISROOT		9

/* Used by base64 routines */
const static char alphabet[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

int checkauth(FILE*, char*, char*);
int errormsg(char*, int);
int base64_decode_line(const char*, char*);
int read_multiline(int, char*, int);
void trim_trailch(char*, char, char, char);


int main(int argc, char* argv[])
{
	FILE* authfile;
	int authenticated = 0;
	char authuser_ok[BUFSIZE];
	int dataphase=0, onheaders=0;
	char buffer[BUFSIZE];
	char* domain = "local";
	char* remotehost;
	char peerIP[BUFSIZE];
	struct sockaddr_in sin;
	struct sockaddr_in sin_peer;
	int sock, sin_len = sizeof(sin_peer);

	if (argc >= 2)
		domain = argv[1];

	if ((getuid()==0) || (geteuid()==0))
		return errormsg(domain, ERROR_UIDISROOT);
		
	remotehost = getenv("REMOTE_HOST");	/* as defined by stunnel */
	if (remotehost)
		strcpy(peerIP, remotehost);
	else if (getpeername(0, (struct sockaddr*)&sin_peer, (socklen_t *)&sin_len) == 0)
		strcpy(peerIP, inet_ntoa(sin_peer.sin_addr));
	else
		strcpy(peerIP, "unknown");

	if (argc != 3)
		return errormsg(domain, ERROR_COMMANDARGS);

	authfile = fopen(AUTHFILE, "r");
	if (authfile == NULL)
                return errormsg(domain, ERROR_NOAUTHFILE);

	sin.sin_family = AF_INET;
	sin.sin_port = htons(25);
	if (inet_aton(argv[2], &sin.sin_addr) == 0)
		return errormsg(domain, ERROR_BADIPADDRESS);
	sock = socket(AF_INET, SOCK_STREAM, 0);
	if (sock == -1)
		return errormsg(domain, ERROR_NOSOCKET);
	if (connect(sock, (struct sockaddr*)&sin, sizeof(sin)) != 0)
	{
		close(sock);
		return errormsg(domain, ERROR_FAILCONNECT);
	}

	/*
		Now connected to actual SMTP server
		Pass through its greeting. Then, while unauthenticated
		respond to HELO with an error
		respond to EHLO with actual SMTP server's reply, tinkered
	*/
	if (read_multiline(sock, buffer, BUFSIZE) != 0)
	{
                close(sock);
		return errormsg(domain, ERROR_LOSTSOCKET);
	}
	printf("%s", buffer);
	fflush(stdout);

	while (fgets(buffer, BUFSIZE, stdin))
	{
		/* QUIT and EHLO are handled at any time except DATA phase */
		if (!dataphase && (strncasecmp(buffer, "QUIT", 4) == 0))
		{
			if (write(sock, buffer, strlen(buffer)) == -1)
			{
				close(sock);
				return errormsg(domain, ERROR_LOSTSOCKET);
			}
			if (read_multiline(sock, buffer, BUFSIZE) != 0)
			{
				close(sock);
				return errormsg(domain, ERROR_LOSTSOCKET);
			}
                        close(sock);
			printf("%s", buffer);
			fflush(stdout);
			return 0;
		}
		else if (!dataphase && (strncasecmp(buffer, "EHLO ", 5) == 0))
		{
			char* token;
			if (write(sock, buffer, strlen(buffer)) == -1)
			{
				close(sock);
				return errormsg(domain, ERROR_LOSTSOCKET);
			}
			if (read_multiline(sock, buffer, BUFSIZE) != 0)
			{
				close(sock);
				return errormsg(domain, ERROR_LOSTSOCKET);
			}
			/* Server sent us a long list of stuff. Insert our own! */
			token = strtok(buffer, "\r\n");
			while (token)
			{
				char sep;
				if ((sscanf(token, "250%c", &sep) == 1) && (sep == ' '))
					printf("250-AUTH PLAIN LOGIN\r\n");
				printf("%s\r\n", token);
				token = strtok(NULL, "\r\n");
			}
			fflush(stdout);
		}
		else if (authenticated)
		{
			if ((dataphase==0) && (strncasecmp(buffer, "DATA", 4)==0))
			{
				dataphase = 1;
				onheaders = 1;
			}
			else if (dataphase)
			{
				dataphase = 2;
				if (onheaders && (*buffer=='\r' || *buffer=='\n'))
				{
					char temp[2*BUFSIZE];
					onheaders = 0;
					sprintf(temp, "X-SMTP-AUTH: %s@%s from [%s]\r\n", authuser_ok, domain, peerIP);
					write(sock, temp, strlen(temp));
				}
				else if ( (strcmp(buffer, ".\r\n")==0) || (strcmp(buffer, ".\n")==0) )
					dataphase = 0;
			}
			if (write(sock, buffer, strlen(buffer)) == -1)
			{
				close(sock);
				return errormsg(domain, ERROR_LOSTSOCKET);
			}
			if (dataphase < 2)
			{
				if (read_multiline(sock, buffer, BUFSIZE) != 0)
				{
					close(sock);
					return errormsg(domain, ERROR_LOSTSOCKET);
				}
				printf("%s", buffer);
				fflush(stdout);
			}
		}
		else
		{
			char decoded[BUFSIZE], authuser[BUFSIZE], authpass[BUFSIZE];
			if (strncasecmp(buffer, "AUTH PLAIN ", 11) == 0)
			{
				memset(decoded, 0, BUFSIZE);
				memset(authuser, 0, BUFSIZE);
				memset(authpass, 0, BUFSIZE);
				trim_trailch(buffer, '\r', '\n', ' ');
				if (base64_decode_line(buffer+11, decoded))
				{
					int userlen = strlen(decoded+1);
					if ((userlen < BUFSIZE) && (strlen(decoded+userlen+2) < BUFSIZE))
					{
						strcpy(authuser, decoded+1);
						strcpy(authpass, decoded+userlen+2);
						openlog("smtpauth", LOG_PID, LOG_MAIL);
						if (checkauth(authfile, authuser, authpass))
						{
							authenticated = 1;
							strcpy(authuser_ok, authuser);
							syslog(LOG_INFO, "Authenticated %s@%s from [%s] using PLAIN",
								authuser_ok, domain, peerIP);
						}
						else
							syslog(LOG_INFO, "Auth failure for %s@%s from [%s] using PLAIN",
								authuser, domain, peerIP);
						closelog();
					}
				}
				if (authenticated)
					printf("235 PLAIN Authentication successful\r\n");
				else
					printf("535 Error: PLAIN authentication failed\r\n");
				fflush(stdout);
			}
			else if (strncasecmp(buffer, "AUTH LOGIN", 10) == 0)
			{
				memset(decoded, 0, BUFSIZE);
				memset(authuser, 0, BUFSIZE);
				memset(authpass, 0, BUFSIZE);
				printf("334 VXNlcm5hbWU6\r\n");	/* Login prompt */
				fflush(stdout);
				if (fgets(buffer, BUFSIZE, stdin))
				{
					trim_trailch(buffer, '\r', '\n', ' ');
					if (base64_decode_line(buffer, authuser))
					{
						printf("334 UGFzc3dvcmQ6\r\n");
						fflush(stdout);
						if (fgets(buffer, BUFSIZE, stdin))
						{
							trim_trailch(buffer, '\r', '\n', ' ');
							if (base64_decode_line(buffer, authpass))
							{
								openlog("smtpauth", LOG_PID, LOG_MAIL);
								if (checkauth(authfile, authuser, authpass))
								{
									authenticated = 1;
									strcpy(authuser_ok, authuser);
									syslog(LOG_INFO, "Authenticated %s@%s from [%s] using LOGIN",
										authuser_ok, domain, peerIP);
								}
								else
									syslog(LOG_INFO, "Auth failure for %s@%s from [%s] using LOGIN",
										authuser, domain, peerIP);
								closelog();
							}
						}
					}
					
				}
				if (authenticated)
					printf("235 LOGIN Authentication successful\r\n");
				else
					printf("535 Error: LOGIN authentication failed\r\n");
				fflush(stdout);
			}
			else
				errormsg(domain, ERROR_NOPERMISSION);
		}
	}

	close(sock);
	return errormsg(domain, ERROR_READFAILURE);	/* client wouldn't see this */
}


/*
        Returns true (1) on success, authenticated
	Returns false (0) on failure

	Note that this function also closes authfile; there is one shot at authentication
*/
int checkauth(FILE* authfile, char* authuser, char* authpass)
{
        char linebuf[BUFSIZE], user[BUFSIZE], pass[BUFSIZE];
	int auth = 0;
	if (authfile)
	{
		while (!auth && (fgets(linebuf, BUFSIZE, authfile)))
		{
			if ((*linebuf == '\n') || (*linebuf == '#'))
				continue;
			else if (sscanf(linebuf, "%s %s\n", user, pass) == 2)
			{
				if ((strcmp(user, authuser) == 0) && (strcmp(pass, authpass) == 0))
					auth = 1;
			}
		}
		fclose(authfile);
		authfile = NULL;
	}
	return auth;
}


int errormsg(char* domain, int code)
{
        switch (code)
	{
		case ERROR_NOPERMISSION:
                        printf("550 Unauthenticated access denied\r\n");
			break;

		case ERROR_LOSTSOCKET:
			printf("421 %s Service not available, closing transmission channel\r\n", domain);
			break;

		default:
			printf("554 SMTP not available (%d)\r\n", code);
	}
	fflush(stdout);
	return 0;
}


/*
	Returns number of bytes written to dest, if this was valid base64 encoded data
	Returns 0 if invalid format
	Can decode non-text. Caller should completely clear dest
*/
int base64_decode_line(const char* input, char* dest)
{
	char outbytes[3];
	int len = strlen(input);	/* input is plain text */
	const char* endpoint = input + len;
	int outpos = 0;		/* required for binary output */

	if (len % 4 != 0)
		return 0;	/* quit, invalid format */
	while (input < endpoint)
	{
		int i, j, towrite=3;
		for (i=2; i<4; i++)	/* count deductions */
			if (input[i] == '=') towrite--;
		for (j=0; j<towrite; j++)
		{
			char *lookup1 = strchr(alphabet, input[j]);
			char *lookup2 = strchr(alphabet, input[j+1]);
			if (!lookup1 || !lookup2)
				return 0;	/* invalid situation */
			outbytes[j] = (((lookup1-alphabet) << (2+j*2)) | ((lookup2 - alphabet) >> (4-2*j)));
		}
		memcpy(dest+outpos, outbytes, towrite);
		outpos += towrite;
		input += 4;
	}
	return outpos;
}



/*
	Reads a single (or multiple) lines from the socket, until
	the buffer contains the sequence "%d <any>\r\n" where the
	start is either at the front of the buffer, or after an \r\n

        space is the total size of the buffer (MUST be > 0)
	Returns 0 on success, nonzero on failure
*/
int read_multiline(int sock, char* buffer, int space)
{
	char* bufpos = buffer;
	memset(buffer, 0, space--);
	while (space > 0)
	{
		char* startpoint = buffer;
		int got = read(sock, bufpos, space);
		if (got == 0)
			return 1;	/* end of stream */
		bufpos += got;
		space -= got;
		/* Look in entire buffer for last-line sequence */
		while (startpoint)
		{
			int value;
			char sep;
			if (startpoint != buffer)
				startpoint++;
			if ((sscanf(startpoint, "%d%c", &value, &sep) == 2) && (sep == ' '))
				return 0;	/* found */
			startpoint = strchr(startpoint+1, '\n');
		}
	}
	return 1;	/* failure, didn't find our pattern */
}



/*
	Trims specified trailing character from a line
*/
void trim_trailch(char* buf, char ch1, char ch2, char ch3)
{
	char* end = buf+strlen(buf)-1;
	while (end>=buf && (*end==ch1 || *end==ch2 || *end==ch3))
	{
		*end = 0;
		end--;
	}
}
