/*
  grepcidr 1.2 - Filter IP addresses matching IPv4 CIDR specification
  Copyright (C) 2004  Jem E. Berkes <jberkes@pc-tools.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "cidr.h"
#include "getopt.h"

#define TXT_VERSION	"grepcidr 1.2\nCopyright (C) 2004  Jem E. Berkes <jberkes@pc-tools.net>\n"
#define TXT_USAGE	"Usage: grepcidr [-V] [-c] [-v] [-e PATTERN | -f FILE]\n"
#define MAXFIELD	512
#define TOKEN_SEPS	"\t ,\r\n"


static char shortopts[] = "ce:f:vV";
char* filename = NULL;
int counting = 0;			/* when non-zero, counts matches */
int invert = 0;				/* flag for inverted mode */
int total_patterns = 0;			/* total CIDR patterns */
int pattern_count[8];			/* pattern counts in each bucket */
struct cidr_spec* bucket[8];		/* pattern buckets */


/*
	main returns 0 on success
	1 on failure
*/
int main(int argc, char* argv[])
{
	unsigned int bucketno;
	char* patbuf = NULL;
	char line[MAXFIELD];
	int foundopt;

	if (argc == 1)
	{
		fprintf(stderr, TXT_USAGE);
		return 1;
	}

	while ((foundopt = getopt(argc, argv, shortopts)) != -1)
	{
		switch (foundopt)
		{
			case 'V':
				puts(TXT_VERSION);
				return 0;
				
			case 'c':
				counting = 1;
				break;
				
			case 'v':
				invert = 1;
				break;
				
			case 'e':
				patbuf = optarg;
				break;

			case 'f':
				filename = optarg;
				break;
				
			default:
				fprintf(stderr, TXT_USAGE);
				return 1;
		}
	}
	
	if (!(patbuf || filename))
	{
		fprintf(stderr, "Specify either -e PATTERN or -f FILE\n");
		return 1;
	}
	
	if (filename)
	/* Loading patterns from file */
	{
		FILE* data = fopen(filename, "r");
		if (data)
		{
			long filesize = 0;
			if (fseek(data, 0, SEEK_END) != 0)
			{
				perror(filename);
				return 1;
			}
			filesize = ftell(data);
			rewind(data);
			if (filesize > 0)
			{
				patbuf = calloc(1, filesize + 1);
				while (fgets(line, sizeof(line), data))
				{
					if ((*line != '#') && (*line != '\n'))
						strcat(patbuf, line);
				}
			}
			else
			{
				fprintf(stderr, "No file data\n");
				return 1;
			}
			fclose(data);
		}
		else
		{
			perror(filename);
			return 1;
		}
	}
	
	/* Fill elements from big list (patbuf) into appropriate buckets */
	{
		char *token, *entire = malloc(1 + strlen(patbuf));
		strcpy(entire, patbuf);
		if (!filename)
		{
			/* Can't modify argv, so make new buffer */
			patbuf = malloc(1 + strlen(entire));
			strcpy(patbuf, entire);
		}
		token = strtok(patbuf, TOKEN_SEPS);
		while (token)
		{
			total_patterns++;
			token = strtok(NULL, TOKEN_SEPS);
		}
		free(patbuf);
		for (bucketno=0; bucketno<8; bucketno++)
			bucket[bucketno] = calloc(total_patterns, sizeof(struct cidr_spec));
		token = strtok(entire, TOKEN_SEPS);
		while (token)
		{
			struct cidr_spec spec;
			if (!cidr_parse(token, &spec))
				fprintf(stderr, "Invalid CIDR %s\n", token);
			else
			{
				struct cidr_spec* patterns;
				if (spec.mask_32 < 0xE0000000)
				{
					/* Rare, must put in all buckets */
					for (bucketno=0; bucketno<8; bucketno++)
					{
						patterns = bucket[bucketno];
						patterns[pattern_count[bucketno]++] = spec;
					}
				}
				else
				{
					/* Determine and place in appropriate bucket*/
					bucketno = (spec.cidr_32 >> 29) & 0x00000007;
					patterns = bucket[bucketno];
					patterns[pattern_count[bucketno]++] = spec;
				}
			}
			token = strtok(NULL, TOKEN_SEPS);
		}
		free(entire);
	}
	
	/* Do line matching */
	while (fgets(line, sizeof(line), stdin))
	{
		unsigned int IP_32 = ip_to_int(line);
		bucketno = (IP_32 >> 29) & 0x00000007;
		if (IP_32)
		{
			int curpat, match=0;
			struct cidr_spec* patterns = bucket[bucketno];

			for (curpat=0; (curpat<pattern_count[bucketno]) && !match; curpat++)
				match = ((IP_32 & patterns[curpat].mask_32) == patterns[curpat].cidr_32);
			if (invert ^ match)
			{
				if (counting)
					counting++;
				else
					printf("%s", line);
			}
		}
	}

	if (counting)
		printf("%u\n", counting-1);
	for (bucketno=0; bucketno<8; bucketno++)
		free(bucket[bucketno]);
	return 0;	
}
