/*
	Functions to parse and compare Classless Inter-Domain Routing (CIDR)
	cidr_parse, cidr_match 1.1
	
	Copyright (C) 2004, Jem E. Berkes <jberkes@pc-tools.net>
	This code is part of http://www.pc-tools.net/unix/grepcidr/
*/

#include <stdio.h>
#include <string.h>
#include "cidr.h"


/*
	Given string, fills in the struct cidr_spec (must be allocated)
	Returns true (nonzero) on success, false (zero) on invalid CIDR
*/
int cidr_parse(const char* cidr, struct cidr_spec* spec)
{
	unsigned int CIDR[4];	/* 4 octets from CIDR specification */
	int maskbits = 32;	/* to compare: between 1 and 32 bits */
	
	/* Sanity checking */
	if (!cidr || !spec)
		return 0;
		
	/* Parse the CIDR specification */
	if (sscanf(cidr, "%u.%u.%u.%u/%d", &CIDR[0], &CIDR[1], &CIDR[2], &CIDR[3], &maskbits) != 5)
	{
		if (sscanf(cidr, "%u.%u.%u.%u", &CIDR[0], &CIDR[1], &CIDR[2], &CIDR[3]) != 4)
			return 0;	/* invalid CIDR */
	}
	if ((maskbits < 1) || (maskbits > 32))
		return 0;	/* invalid CIDR */
	if ((CIDR[0] > 255) || (CIDR[1] > 255) || (CIDR[2] > 255) || (CIDR[3] > 255))
		return 0;	/* invalid CIDR */

	spec->mask_32 = ~((1 << (32-maskbits))-1) & 0xFFFFFFFF;	/* requires 32-bit int or larger */
	spec->cidr_32 = ((CIDR[0] << 24) | (CIDR[1] << 16) | (CIDR[2] << 8) | CIDR[3]) & spec->mask_32;
	
	return 1;	/* success */
}


/*
	Compare IP address against CIDR specification
	Returns true (nonzero) on match, false (zero) otherwise
*/
int cidr_match(const char* ip, struct cidr_spec* spec)
{
	if (!ip || !spec)
		return 0;
	/* See if IP matches */
	return ((ip_to_int(ip) & spec->mask_32) == spec->cidr_32);
}



/*
	Convert IP address string to 32-bit integer version
	Returns 0 on failure
*/
unsigned int ip_to_int(const char* ip)
{
	unsigned int IP[4];	/* 4 octets for IP address */

	if (!ip)
		return 0;
	/* Parse the supplied IP address */
	if (sscanf(ip, "%u.%u.%u.%u", &IP[0], &IP[1], &IP[2], &IP[3]) != 4)
		return 0;	/* invalid IP */
	if ((IP[0] > 255) || (IP[1] > 255) || (IP[2] > 255) || (IP[3] > 255))
		return 0;	/* invalid IP */
	return (IP[0] << 24) | (IP[1] << 16) | (IP[2] << 8) | IP[3];
}
