/*
  grepcidr 1.0 - Filter IP addresses matching IPv4 CIDR specification
  Copyright (C) 2004  Jem E. Berkes <jberkes@pc-tools.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <memory.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "cidr.h"
#include "getopt.h"

#define TXT_VERSION	"grepcidr 1.0, by Jem Berkes <jberkes@pc-tools.net>"
#define TXT_USAGE	"Usage: grepcidr [-V] [-v] [-e PATTERN | -f FILE]\n"
#define MAXFIELD	512
#define TOKEN_SEPS	"\t ,\r\n"


int invert = 0;
int patcount = 0;		/* number of elements in patterns */
char** patterns = NULL;		/* pointers into patbuf */
char* patbuf = NULL;		/* actually stores all patterns */
char* filename = NULL;
static char shortopts[] = "e:f:vV";


/*
	main returns 0 on success
	1 on failure
*/
int main(int argc, char* argv[])
{
	char line[MAXFIELD];
	int foundopt;

	if (argc == 1)
	{
		fprintf(stderr, TXT_USAGE);
		return 1;
	}

	while ((foundopt = getopt(argc, argv, shortopts)) != -1)
	{
		switch (foundopt)
		{
			case 'V':
				puts(TXT_VERSION);
				return 0;
				
			case 'v':
				invert = 1;
				break;
				
			case 'e':
				patbuf = optarg;
				break;

			case 'f':
				filename = optarg;
				break;
				
			default:
				fprintf(stderr, TXT_USAGE);
				return 1;
		}
	}
	
	if (!(patbuf || filename))
	{
		fprintf(stderr, "Specify either -e PATTERN or -f FILE\n");
		return 1;
	}
	
	if (filename)
	/* Loading patterns from file */
	{
		FILE* data = fopen(filename, "r");
		if (data)
		{
			long filesize = 0;
			if (fseek(data, 0, SEEK_END) != 0)
			{
				perror(filename);
				return 1;
			}
			filesize = ftell(data);
			rewind(data);
			if (filesize > 0)
			{
				patbuf = calloc(1, filesize + 1);
				while (fgets(line, sizeof(line), data))
				{
					if ((*line != '#') && (*line != '\n'))
						strcat(patbuf, line);
				}
			}
			else
			{
				fprintf(stderr, "No file data\n");
				return 1;
			}
			fclose(data);
		}
		else
		{
			perror(filename);
			return 1;
		}
	}
	
	/* Convert big list in patbuf to indexed list, via tokenized patterns */
	{
		char *entire, *token;
		int curpat = 0;
		
		entire = malloc(1 + strlen(patbuf));
		strcpy(entire, patbuf);
		if (filename)
			free(patbuf);
		patbuf = malloc(1 + strlen(entire));
		strcpy(patbuf, entire);
		token = strtok(patbuf, TOKEN_SEPS);
		while (token)
		{
			patcount++;
			token = strtok(NULL, TOKEN_SEPS);
		}
		strcpy(patbuf, entire);
		free(entire);
		patterns = calloc(patcount, sizeof(char*));
		token = strtok(patbuf, TOKEN_SEPS);
		while (token)
		{
			patterns[curpat++] = token;
			token = strtok(NULL, TOKEN_SEPS);
		}
		/* patbuf remains allocated (storing text pointed to by patterns) */
	}


	/* Do line matching */
	while (fgets(line, sizeof(line), stdin))
	{
		int curpat, match=0;
		for (curpat=0; (curpat<patcount) && !match; curpat++)
			match = cidr_match(line, patterns[curpat]);

		if (invert ^ match)
			printf("%s", line);
	}

	free(patterns);
	free(patbuf);
	return 0;	
}
